// Flluir Premiere Plugin - Main JavaScript (CEP)
(function() {
  'use strict';

  // CSInterface initialization
  var csInterface = null;
  var extensionId = null;

  // State
  var state = {
    apiUrl: '',
    authToken: '',
    currentFile: null,
    projects: [],
    files: [],
    selectedProjectId: null,
    selectedFile: null,
    comments: [],
    isConnected: false
  };

  // DOM Elements (será preenchido na inicialização)
  var elements = {};
  
  // Função para buscar elementos do DOM de forma segura
  function bindElements() {
    try {
      elements = {
        statusIndicator: document.getElementById('statusIndicator'),
        apiUrl: document.getElementById('apiUrl'),
        authToken: document.getElementById('authToken'),
        testConnection: document.getElementById('testConnection'),
        saveConfig: document.getElementById('saveConfig'),
        currentFile: document.getElementById('currentFile'),
        projectsList: document.getElementById('projectsList'),
        filesList: document.getElementById('filesList'),
        refreshProjects: document.getElementById('refreshProjects'),
        refreshFiles: document.getElementById('refreshFiles'),
        openFile: document.getElementById('openFile'),
        videoPreview: document.getElementById('videoPreview'),
        videoPreviewContainer: document.getElementById('videoPreviewContainer'),
        videoEmpty: document.getElementById('videoEmpty'),
        commentsList: document.getElementById('commentsList'),
        loadComments: document.getElementById('loadComments'),
        syncToPremiere: document.getElementById('syncToPremiere'),
        clearMarkers: document.getElementById('clearMarkers'),
        debugConsole: document.getElementById('debugConsole'),
        toggleDebug: document.getElementById('toggleDebug')
      };
      
      // Verificar elementos críticos
      var criticalElements = ['statusIndicator', 'apiUrl', 'authToken', 'testConnection', 'saveConfig'];
      var missingElements = [];
      for (var i = 0; i < criticalElements.length; i++) {
        if (!elements[criticalElements[i]]) {
          missingElements.push(criticalElements[i]);
        }
      }
      
      if (missingElements.length > 0) {
        console.error('Elementos críticos não encontrados:', missingElements);
        return false;
      }
      
      console.log('Todos os elementos do DOM foram vinculados com sucesso');
      return true;
    } catch (error) {
      console.error('Erro ao vincular elementos do DOM:', error);
      return false;
    }
  }

  // Initialize CSInterface
  function initCSInterface() {
    try {
      if (typeof CSInterface === 'undefined') {
        console.error('CSInterface não está definido. Verifique se CSInterface.js está presente e carregou corretamente.');
        return false;
      }
      csInterface = new CSInterface();
      extensionId = csInterface.getExtensionID();
      logDebug('CSInterface inicializado com sucesso. Extension ID: ' + extensionId, 'success');
      return true;
    } catch (error) {
      console.error('Erro ao inicializar CSInterface:', error);
      logDebug('Erro ao inicializar CSInterface: ' + error.message, 'error');
      return false;
    }
  }

  // Load saved configuration
  function loadConfig() {
    try {
      var savedApiUrl = localStorage.getItem('flluir_api_url');
      var savedToken = localStorage.getItem('flluir_auth_token');
      
      if (savedApiUrl) {
        state.apiUrl = savedApiUrl;
        if (elements.apiUrl) elements.apiUrl.value = savedApiUrl;
      }
      
      if (savedToken) {
        state.authToken = savedToken;
        elements.authToken.value = savedToken;
      }
      
      logDebug('Configuração carregada do localStorage', 'info');
    } catch (error) {
      logDebug('Erro ao carregar configuração: ' + error.message, 'error');
    }
  }

  // Save configuration
  function saveConfig() {
    try {
      state.apiUrl = elements.apiUrl ? elements.apiUrl.value.trim() : '';
      state.authToken = elements.authToken.value.trim();
      
      localStorage.setItem('flluir_api_url', state.apiUrl);
      localStorage.setItem('flluir_auth_token', state.authToken);
      
      logDebug('Configuração salva com sucesso', 'success');
      showMessage('Configuração salva!', 'success');
      
      // Auto fluxo após salvar
      if (state.apiUrl && state.authToken) {
        testConnection().then(function() {
          if (state.isConnected) {
            loadProjects({ autoSelectFirst: true });
          }
        }).catch(function() {});
      }
    } catch (error) {
      logDebug('Erro ao salvar configuração: ' + error.message, 'error');
      showMessage('Erro ao salvar configuração', 'error');
    }
  }

  // Read error message from response
  function readErrorMessage(response) {
    return response.text().then(function(text) {
      // 403 com HTML de challenge/proteção (evitar colar HTML inteiro na mensagem)
      if (response.status === 403 && text && text.length > 200 && text.indexOf('<!DOCTYPE html>') !== -1) {
        return 'Requisição bloqueada (possível proteção de segurança do servidor). Verifique URL e token.';
      }
      try {
        var data = JSON.parse(text);
        return data.error || text || response.status + ' ' + response.statusText;
      } catch (e) {
        // Resposta não é JSON: se for HTML longo, não colar no erro
        if (text && text.length > 300 && (text.indexOf('<') !== -1 && text.indexOf('>') !== -1)) {
          return response.status + ' – servidor retornou HTML em vez de JSON (possível bloqueio de segurança/proxy).';
        }
        return text || response.status + ' ' + response.statusText;
      }
    }).catch(function() {
      return response.status + ' ' + response.statusText;
    });
  }

  // Fetch JSON helper
  function fetchJSON(url, options) {
    return fetch(url, options).then(function(response) {
      if (!response.ok) {
        return readErrorMessage(response).then(function(msg) {
          throw new Error('Erro ' + response.status + ': ' + msg);
        });
      }
      return response.json();
    });
  }

  // Test API connection
  function testConnection() {
    if (!state.apiUrl || !state.authToken) {
      showMessage('Configure a URL da API e o token antes de testar', 'error');
      return Promise.reject(new Error('Configuração incompleta'));
    }

    updateStatus('connecting');
    elements.testConnection.disabled = true;

    return fetchJSON(state.apiUrl.replace(/\/$/, '') + '/api/premiere/ping', {
      method: 'GET',
      headers: {
        'Authorization': 'Bearer ' + state.authToken,
        'Content-Type': 'application/json'
      }
    }).then(function(data) {
      updateStatus('connected');
      var mode = data.mode ? ' (modo: ' + data.mode + ')' : '';
      showMessage('Conexão estabelecida com sucesso!' + mode, 'success');
      logDebug('Conexão testada com sucesso' + mode, 'success');
      
      // Auto carregar projetos após conectar
      loadProjects({ autoSelectFirst: true });
    }).catch(function(error) {
      updateStatus('error');
      showMessage('Erro ao conectar: ' + error.message, 'error');
      logDebug('Erro ao testar conexão: ' + error.message, 'error');
      throw error;
    }).finally(function() {
      elements.testConnection.disabled = false;
    });
  }

  // Get active sequence info from Premiere
  function getActiveSequenceInfo() {
    return new Promise(function(resolve, reject) {
      if (!csInterface) {
        reject(new Error('CSInterface não inicializado'));
        return;
      }

      var script = 'if (typeof $.global.FlluirSync !== "undefined") { ' +
                   '  $.global.FlluirSync.getSequenceInfo(); ' +
                   '} else { ' +
                   '  JSON.stringify({error: "FlluirSync não carregado"}); ' +
                   '}';

      csInterface.evalScript(script, function(result) {
        try {
          var data = JSON.parse(result);
          if (data.error) {
            reject(new Error(data.error));
          } else {
            resolve(data);
          }
        } catch (error) {
          reject(new Error('Erro ao parsear resposta: ' + error.message));
        }
      });
    });
  }

  // Load projects
  function loadProjects(opts) {
    opts = opts || {};
    if (!state.apiUrl || !state.authToken) {
      showMessage('Configure a URL da API e o token primeiro', 'error');
      return Promise.resolve();
    }

    elements.refreshProjects.disabled = true;

    return fetchJSON(state.apiUrl.replace(/\/$/, '') + '/api/premiere/projects', {
      method: 'GET',
      headers: { 'Authorization': 'Bearer ' + state.authToken }
    }).then(function(data) {
      state.projects = data.projects || [];
      renderProjects();
      logDebug('Projetos carregados: ' + state.projects.length, 'success');

      // Auto: selecionar o primeiro projeto
      var autoSelectFirst = !!opts.autoSelectFirst;
      if (autoSelectFirst && !state.selectedProjectId && state.projects.length > 0) {
        selectProject(state.projects[0].id, { autoSelectFirstFile: true, autoLoadComments: true });
      } else if (state.selectedProjectId) {
        var exists = state.projects.some(function(p) { return p.id === state.selectedProjectId; });
        if (!exists) {
          state.selectedProjectId = null;
          state.selectedFile = null;
          state.comments = [];
          renderFiles();
          renderComments([]);
        }
      }
    }).catch(function(err) {
      showMessage('Erro ao carregar projetos: ' + err.message, 'error');
      logDebug('Erro ao carregar projetos: ' + err.message, 'error');
    }).finally(function() {
      elements.refreshProjects.disabled = false;
    });
  }

  // Load files
  function loadFiles(projectId, opts) {
    opts = opts || {};
    if (!state.apiUrl || !state.authToken) {
      showMessage('Configure a URL da API e o token primeiro', 'error');
      return Promise.resolve();
    }

    elements.refreshFiles.disabled = true;

    return fetchJSON(state.apiUrl.replace(/\/$/, '') + '/api/premiere/files?projectId=' + encodeURIComponent(projectId), {
      method: 'GET',
      headers: { 'Authorization': 'Bearer ' + state.authToken }
    }).then(function(data) {
      var files = data.files || [];
      logDebug('Arquivos carregados: ' + files.length, 'success');

      // Para cada arquivo, tentar buscar uma thumbnail assinada (se existir)
      var thumbPromises = files.map(function (f) {
        return fetchJSON(state.apiUrl.replace(/\/$/, '') + '/api/premiere/file-signed-url?fileId=' + encodeURIComponent(f.id), {
          method: 'GET',
          headers: { 'Authorization': 'Bearer ' + state.authToken }
        }).then(function (res) {
          if (res.thumbnailSignedUrl) {
            f.thumbnail_url = res.thumbnailSignedUrl;
          } else if (!f.thumbnail_url && res.signedUrl) {
            // Fallback extremo: usar o próprio vídeo como thumbnail
            f.thumbnail_url = res.signedUrl;
          }
        }).catch(function () {
          // Se falhar para um arquivo, apenas segue sem thumbnail
        });
      });

      return Promise.all(thumbPromises).then(function () {
        state.files = files;
        renderFiles();

        var autoSelectFirst = !!opts.autoSelectFirst;
        if (autoSelectFirst && !state.selectedFile && state.files.length > 0) {
          selectFile(state.files[0].id, { autoLoadComments: !!opts.autoLoadComments });
        } else if (state.selectedFile) {
          var stillExists = state.files.some(function (f) { return f.id === state.selectedFile.id; });
          if (!stillExists) {
            state.selectedFile = null;
            state.comments = [];
            renderComments([]);
          }
        }
      });
    }).catch(function(err) {
      showMessage('Erro ao carregar arquivos: ' + err.message, 'error');
      logDebug('Erro ao carregar arquivos: ' + err.message, 'error');
    }).finally(function() {
      elements.refreshFiles.disabled = false;
    });
  }

  // Select project
  function selectProject(projectId, opts) {
    opts = opts || {};
    state.selectedProjectId = projectId;
    state.selectedFile = null;
    state.comments = [];
    renderFiles();
    renderComments([]);

    elements.refreshFiles.disabled = false;
    elements.openFile.disabled = true;
    elements.loadComments.disabled = true;
    elements.syncToPremiere.disabled = true;

    loadFiles(projectId, { autoSelectFirst: !!opts.autoSelectFirstFile, autoLoadComments: !!opts.autoLoadComments });
  }

  // Select file
  function selectFile(fileId, opts) {
    opts = opts || {};
    state.selectedFile = state.files.find(function(f) { return f.id === fileId; });
    if (!state.selectedFile) return;

    elements.openFile.disabled = false;
    elements.loadComments.disabled = false;

    // Preview do vídeo (não bloqueia se falhar)
    loadVideoPreview(state.selectedFile.id).catch(function(err) {
      logDebug('Preview não carregado (não crítico): ' + String(err.message || err), 'error');
    });

    if (opts.autoLoadComments) {
      loadComments();
    }
  }

  // Render projects
  function renderProjects() {
    if (!elements.projectsList) return;
    if (!state.projects || state.projects.length === 0) {
      elements.projectsList.innerHTML = '<div class="empty-state"><div class="empty-state-text">Nenhum projeto encontrado</div></div>';
      return;
    }

    var html = '';
    state.projects.forEach(function(p) {
      var subtitle = (p.client_name ? p.client_name + ' • ' : '') + ('#' + (p.project_number || '—'));
      html += '<div class="list-item" data-id="' + p.id + '">' +
              '<div class="list-item-title">' + (p.name || 'Projeto') + '</div>' +
              '<div class="list-item-subtitle">' + subtitle + '</div>' +
              '</div>';
    });

    elements.projectsList.innerHTML = html;

    var items = elements.projectsList.querySelectorAll('.list-item');
    items.forEach(function(item) {
      if (state.selectedProjectId && item.dataset.id === state.selectedProjectId) {
        item.classList.add('active');
      }

      item.addEventListener('click', function() {
        items.forEach(function(i) { i.classList.remove('active'); });
        item.classList.add('active');
        selectProject(item.dataset.id, { autoSelectFirstFile: true, autoLoadComments: true });
      });
    });
  }

  // Render files
  function renderFiles() {
    if (!elements.filesList) return;
    if (!state.selectedProjectId) {
      elements.filesList.innerHTML = '<div class="empty-state"><div class="empty-state-text">Selecione um projeto</div></div>';
      return;
    }
    if (!state.files || state.files.length === 0) {
      elements.filesList.innerHTML = '<div class="empty-state"><div class="empty-state-text">Nenhum arquivo no projeto</div></div>';
      return;
    }

    var html = '';
    state.files.forEach(function(f) {
      var subtitle = (f.type || 'vídeo') + (f.duration ? (' • ' + Math.round(f.duration) + 's') : '');
      var isActive = state.selectedFile && state.selectedFile.id === f.id;
      var safeName = (f.name || f.original_name || 'Vídeo').replace(/"/g, '&quot;');

      // Montar URL da thumbnail (aceita absoluta ou relativa)
      var thumb = f.thumbnail_url || '';
      if (thumb) {
        var lower = thumb.toLowerCase();
        var isAbsolute = lower.indexOf('http://') === 0 || lower.indexOf('https://') === 0;
        if (!isAbsolute && state.apiUrl) {
          var base = state.apiUrl.replace(/\/$/, '');
          // remover barra final do base
          if (base.charAt(base.length - 1) === '/') {
            base = base.slice(0, -1);
          }
          // garantir que o caminho da thumb começa com /
          if (thumb.charAt(0) !== '/') {
            thumb = '/' + thumb;
          }
          thumb = base + thumb;
        }
      }

      html += ''
        + '<div class="video-thumb' + (isActive ? ' active' : '') + '" data-id="' + f.id + '">'
        + '  <div class="video-thumb-image">';
      if (thumb) {
        html += '    <img src="' + thumb + '" alt="' + safeName + '">';
      } else {
        html += '    <div class="video-thumb-placeholder">Sem miniatura</div>';
      }
      html += '  </div>'
        + '  <div class="video-thumb-title" title="' + safeName + '">' + safeName + '</div>'
        + '  <div class="video-thumb-meta">' + subtitle + '</div>'
        + '</div>';
    });
    elements.filesList.innerHTML = html;

    var items = elements.filesList.querySelectorAll('.video-thumb');
    items.forEach(function(item) {
      item.addEventListener('click', function() {
        items.forEach(function(i) { i.classList.remove('active'); });
        item.classList.add('active');
        selectFile(item.dataset.id, { autoLoadComments: true });
      });
    });
  }

  // Reset video preview
  function resetVideoPreview() {
    if (!elements.videoPreview) return;
    try {
      elements.videoPreview.pause();
    } catch (e) {}
    try {
      elements.videoPreview.removeAttribute('src');
      elements.videoPreview.load();
    } catch (e) {}
    elements.videoPreview.style.display = 'none';
    if (elements.videoEmpty) {
      elements.videoEmpty.style.display = 'block';
      elements.videoEmpty.textContent = 'Selecione um arquivo de vídeo para visualizar';
    }
  }

  // Load video preview
  function loadVideoPreview(fileId) {
    if (!elements.videoPreview) return Promise.resolve();

    if (!state.apiUrl || !state.authToken || !fileId) {
      resetVideoPreview();
      return Promise.resolve();
    }

    if (elements.videoEmpty) {
      elements.videoEmpty.style.display = 'block';
      elements.videoEmpty.textContent = 'Carregando preview...';
    }
    elements.videoPreview.style.display = 'none';

    return fetchJSON(state.apiUrl.replace(/\/$/, '') + '/api/premiere/file-signed-url?fileId=' + encodeURIComponent(fileId), {
      method: 'GET',
      headers: { 'Authorization': 'Bearer ' + state.authToken }
    }).then(function(data) {
      var signedUrl = data.signedUrl || null;
      if (!signedUrl) throw new Error('Resposta sem signedUrl');

      logDebug('URL assinada obtida: ' + signedUrl.substring(0, 100) + '...', 'info');

      elements.videoPreview.src = signedUrl;
      elements.videoPreview.style.display = 'block';
      if (elements.videoEmpty) elements.videoEmpty.style.display = 'none';

      return new Promise(function(resolve, reject) {
        var timeout = setTimeout(function() {
          reject(new Error('Timeout ao carregar vídeo (30s)'));
        }, 30000);

        var onLoaded = function() {
          clearTimeout(timeout);
          elements.videoPreview.removeEventListener('loadedmetadata', onLoaded);
          elements.videoPreview.removeEventListener('error', onError);
          logDebug('Vídeo carregado: ' + (elements.videoPreview.duration || 'N/A') + 's', 'success');
          resolve();
        };

        var onError = function(e) {
          clearTimeout(timeout);
          elements.videoPreview.removeEventListener('loadedmetadata', onLoaded);
          elements.videoPreview.removeEventListener('error', onError);
          logDebug('Erro ao carregar vídeo: ' + (e.message || 'Erro desconhecido'), 'error');
          reject(new Error('Erro ao carregar vídeo'));
        };

        elements.videoPreview.addEventListener('loadedmetadata', onLoaded);
        elements.videoPreview.addEventListener('error', onError);
        
        if (elements.videoPreview.readyState >= 2) {
          onLoaded();
        }
      }).then(function() {
        elements.videoPreview.ontimeupdate = function() {
          syncActiveCommentWithTime(elements.videoPreview.currentTime || 0);
        };
        logDebug('Preview do vídeo configurado com sucesso', 'success');
      });
    }).catch(function(err) {
      resetVideoPreview();
      if (elements.videoEmpty) {
        elements.videoEmpty.style.display = 'block';
        elements.videoEmpty.textContent = 'Não foi possível carregar o preview do vídeo.';
      }
      logDebug('Erro no preview do vídeo: ' + String(err.message || err), 'error');
      throw err;
    });
  }

  // Sync active comment with video time
  function syncActiveCommentWithTime(currentTimeSeconds) {
    if (!state.comments || state.comments.length === 0) return;
    if (!elements.commentsList) return;

    var activeId = null;
    for (var i = 0; i < state.comments.length; i++) {
      var c = state.comments[i];
      var t = Number(c.timestamp || 0);
      if (!isFinite(t)) continue;
      if (t <= currentTimeSeconds + 0.05) activeId = c.id;
      else break;
    }

    var nodes = elements.commentsList.querySelectorAll('.comment-item');
    nodes.forEach(function(n) {
      if (activeId && n.dataset.id === String(activeId)) n.classList.add('active');
      else n.classList.remove('active');
    });
  }

  // Load comments from API
  function loadComments() {
    if (!state.apiUrl || !state.authToken) {
      showMessage('Configure a URL da API e o token primeiro', 'error');
      return;
    }

    var fileId = state.selectedFile ? state.selectedFile.id : null;
    if (!fileId) {
      showMessage('Selecione um arquivo primeiro', 'error');
      return;
    }

    elements.loadComments.disabled = true;

    fetchJSON(state.apiUrl.replace(/\/$/, '') + '/api/premiere/sync?fileId=' + encodeURIComponent(fileId), {
      method: 'GET',
      headers: {
        'Authorization': 'Bearer ' + state.authToken,
        'Content-Type': 'application/json'
      }
    }).then(function(data) {
      state.comments = data.markers || [];
      
      logDebug('Comentários recebidos da API: ' + state.comments.length, 'info');
      if (state.comments.length > 0) {
        logDebug('Primeiro comentário: ID=' + state.comments[0].id + ', timestamp=' + state.comments[0].timestamp, 'info');
      }
      
      renderComments(state.comments);
      logDebug('Comentários carregados: ' + state.comments.length, 'success');

      elements.loadComments.disabled = false;
      elements.syncToPremiere.disabled = state.comments.length === 0;

      // Highlight inicial baseado no tempo atual do preview
      if (elements.videoPreview && elements.videoPreview.style.display !== 'none') {
        syncActiveCommentWithTime(elements.videoPreview.currentTime || 0);
      }
    }).catch(function(error) {
      showMessage('Erro ao carregar comentários: ' + error.message, 'error');
      logDebug('Erro ao carregar comentários: ' + error.message, 'error');
      elements.loadComments.disabled = false;
    });
  }

  // Sync comments to Premiere markers
  function syncToPremiere() {
    if (state.comments.length === 0) {
      showMessage('Nenhum comentário para sincronizar', 'error');
      return;
    }

    if (!csInterface) {
      showMessage('CSInterface não inicializado', 'error');
      return;
    }

    logDebug('=== INICIANDO SINCRONIZAÇÃO ===', 'info');
    logDebug('Total de comentários: ' + state.comments.length, 'info');
    
    // #region agent log
    fetch('http://127.0.0.1:7242/ingest/1c8fff22-65c6-4471-824d-a58282891d16',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({location:'main.js:578',message:'syncToPremiere iniciado',data:{commentsCount:state.comments.length},timestamp:Date.now(),sessionId:'debug-session',runId:'run1',hypothesisId:'D'})}).catch(()=>{});
    // #endregion
    
    // Log do primeiro comentário para debug
    if (state.comments.length > 0) {
      var firstComment = state.comments[0];
      logDebug('Primeiro comentário: ID=' + firstComment.id + ', timestamp=' + firstComment.timestamp + ', author=' + firstComment.author, 'info');
    }

      // Primeiro, obter o caminho da extensão no JavaScript (csInterface não está disponível no ExtendScript)
      var extensionPath = '';
      try {
        extensionPath = csInterface.getSystemPath(csInterface.EXTENSION_PATH || 'extension');
      } catch (e1) {
        try {
          extensionPath = csInterface.getSystemPath('extension');
        } catch (e2) {
          // Usar caminho absoluto conhecido como fallback
          extensionPath = '/Users/vitorpassalia/Library/Application Support/Adobe/CEP/extensions/Plugin Flluir';
        }
      }
      
      // Normalizar caminho
      extensionPath = extensionPath.replace(/\\/g, '/').replace(/file:\/\//g, '');
      var jsxPath = extensionPath + '/jsx/FlluirSync.jsx';
      
      // #region agent log
      fetch('http://127.0.0.1:7242/ingest/1c8fff22-65c6-4471-824d-a58282891d16',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({location:'main.js:591',message:'Caminho da extensão obtido',data:{extensionPath:extensionPath,jsxPath:jsxPath},timestamp:Date.now(),sessionId:'debug-session',runId:'run2',hypothesisId:'A'})}).catch(()=>{});
      // #endregion
      
      // Verificar se o JSX está carregado e carregar se necessário
      // Simplificar o script para evitar erros de sintaxe
      var checkScript = 'if (typeof $.global !== "undefined" && typeof $.global.FlluirSync !== "undefined") { ' +
                        '  JSON.stringify({status: "OK", message: "JSX já disponível", flluirSyncExists: true}); ' +
                        '} else { ' +
                        '  var jsxPath = "' + jsxPath.replace(/"/g, '\\"') + '"; ' +
                        '  try { ' +
                        '    $.evalFile(jsxPath); ' +
                        '    if (typeof $.global !== "undefined" && typeof $.global.FlluirSync !== "undefined") { ' +
                        '      JSON.stringify({status: "OK", message: "JSX carregado", flluirSyncExists: true, jsxPath: jsxPath}); ' +
                        '    } else { ' +
                        '      JSON.stringify({status: "ERRO", message: "JSX não disponível após carregar", flluirSyncExists: false, jsxPath: jsxPath}); ' +
                        '    } ' +
                        '  } catch (fileError) { ' +
                        '    JSON.stringify({status: "ERRO", message: "Erro ao carregar JSX", fileError: fileError.toString(), jsxPath: jsxPath}); ' +
                        '  } ' +
                        '}';

    csInterface.evalScript(checkScript, function(checkResult) {
      logDebug('Verificação JSX: ' + checkResult, 'info');
      
      // #region agent log
      fetch('http://127.0.0.1:7242/ingest/1c8fff22-65c6-4471-824d-a58282891d16',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({location:'main.js:627',message:'JSX check raw result',data:{raw:checkResult,length:checkResult?checkResult.length:0},timestamp:Date.now(),sessionId:'debug-session',runId:'run2',hypothesisId:'A'})}).catch(()=>{});
      // #endregion
      
      var parsedResult;
      if (!checkResult || checkResult.trim() === '' || checkResult.indexOf('EvalScript error') === 0) {
        // #region agent log
        fetch('http://127.0.0.1:7242/ingest/1c8fff22-65c6-4471-824d-a58282891d16',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({location:'main.js:630',message:'JSX check evalScript error',data:{raw:checkResult},timestamp:Date.now(),sessionId:'debug-session',runId:'run2',hypothesisId:'A'})}).catch(()=>{});
        // #endregion
        parsedResult = { status: 'ERRO', message: checkResult || 'EvalScript error' };
      } else {
        try {
          parsedResult = JSON.parse(checkResult);
          // #region agent log
          fetch('http://127.0.0.1:7242/ingest/1c8fff22-65c6-4471-824d-a58282891d16',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({location:'main.js:636',message:'JSX check result parsed',data:parsedResult,timestamp:Date.now(),sessionId:'debug-session',runId:'run2',hypothesisId:'A,B,C'})}).catch(()=>{});
          // #endregion
        } catch(e) {
          // #region agent log
          fetch('http://127.0.0.1:7242/ingest/1c8fff22-65c6-4471-824d-a58282891d16',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({location:'main.js:640',message:'JSX check parse error',data:{raw:checkResult,error:e.toString()},timestamp:Date.now(),sessionId:'debug-session',runId:'run2',hypothesisId:'A'})}).catch(()=>{});
          // #endregion
          parsedResult = { status: 'ERRO', message: checkResult };
        }
      }
      
      if (!parsedResult || parsedResult.status === 'ERRO' || (parsedResult.message && parsedResult.message.indexOf('não') !== -1 && parsedResult.message.indexOf('OK') === -1)) {
        showMessage('Erro ao carregar JSX: ' + (parsedResult.message || checkResult), 'error');
        logDebug('JSX não disponível: ' + JSON.stringify(parsedResult), 'error');
        logDebug('Tentando carregar JSX manualmente...', 'info');
        
        // #region agent log
        fetch('http://127.0.0.1:7242/ingest/1c8fff22-65c6-4471-824d-a58282891d16',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({location:'main.js:633',message:'Tentando carregamento manual JSX',data:{previousResult:parsedResult},timestamp:Date.now(),sessionId:'debug-session',runId:'run1',hypothesisId:'E'})}).catch(()=>{});
        // #endregion
        
        // Tentar carregar usando caminho absoluto conhecido (macOS)
        // Simplificar para evitar erros de sintaxe
        var manualLoadScript = 'var jsxPath = "/Users/vitorpassalia/Library/Application Support/Adobe/CEP/extensions/Plugin Flluir/jsx/FlluirSync.jsx"; ' +
                               'try { ' +
                               '  $.evalFile(jsxPath); ' +
                               '  if (typeof $.global !== "undefined" && typeof $.global.FlluirSync !== "undefined") { ' +
                               '    JSON.stringify({status: "OK", message: "JSX carregado manualmente", flluirSyncExists: true, jsxPath: jsxPath}); ' +
                               '  } else { ' +
                               '    JSON.stringify({status: "ERRO", message: "JSX não disponível após carregar manualmente", flluirSyncExists: false, jsxPath: jsxPath}); ' +
                               '  } ' +
                               '} catch (fileError) { ' +
                               '  JSON.stringify({status: "ERRO", message: "Erro ao carregar JSX", fileError: fileError.toString(), jsxPath: jsxPath}); ' +
                               '}';
        
        csInterface.evalScript(manualLoadScript, function(manualResult) {
          logDebug('Resultado carregamento manual: ' + manualResult, 'info');
          
          // #region agent log
          fetch('http://127.0.0.1:7242/ingest/1c8fff22-65c6-4471-824d-a58282891d16',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({location:'main.js:658',message:'JSX manual raw result',data:{raw:manualResult,length:manualResult?manualResult.length:0},timestamp:Date.now(),sessionId:'debug-session',runId:'run2',hypothesisId:'A'})}).catch(()=>{});
          // #endregion
          
          var parsedManual;
          if (!manualResult || manualResult.trim() === '' || manualResult.indexOf('EvalScript error') === 0) {
            // #region agent log
            fetch('http://127.0.0.1:7242/ingest/1c8fff22-65c6-4471-824d-a58282891d16',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({location:'main.js:662',message:'JSX manual evalScript error',data:{raw:manualResult},timestamp:Date.now(),sessionId:'debug-session',runId:'run2',hypothesisId:'A'})}).catch(()=>{});
            // #endregion
            parsedManual = { status: 'ERRO', message: manualResult || 'EvalScript error' };
          } else {
            try {
              parsedManual = JSON.parse(manualResult);
              // #region agent log
              fetch('http://127.0.0.1:7242/ingest/1c8fff22-65c6-4471-824d-a58282891d16',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({location:'main.js:668',message:'JSX manual result parsed',data:parsedManual,timestamp:Date.now(),sessionId:'debug-session',runId:'run2',hypothesisId:'A,B,C,E'})}).catch(()=>{});
              // #endregion
            } catch(e) {
              // #region agent log
              fetch('http://127.0.0.1:7242/ingest/1c8fff22-65c6-4471-824d-a58282891d16',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({location:'main.js:672',message:'JSX manual parse error',data:{raw:manualResult,error:e.toString()},timestamp:Date.now(),sessionId:'debug-session',runId:'run2',hypothesisId:'A'})}).catch(()=>{});
              // #endregion
              parsedManual = { status: 'ERRO', message: manualResult };
            }
          }
          
          if (parsedManual && parsedManual.status === 'OK') {
            logDebug('JSX carregado manualmente com sucesso!', 'success');
            // Continuar com a sincronização
            proceedWithSync();
          } else {
            showMessage('Não foi possível carregar o JSX. Verifique se o arquivo existe.', 'error');
            logDebug('Falha ao carregar JSX manualmente: ' + JSON.stringify(parsedManual), 'error');
          }
        });
        return;
      }
      
      // Se chegou aqui, o JSX está OK
      proceedWithSync();
    });
    
    function proceedWithSync() {
      // #region agent log
      fetch('http://127.0.0.1:7242/ingest/1c8fff22-65c6-4471-824d-a58282891d16',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({location:'main.js:720',message:'proceedWithSync chamado',data:{commentsCount:state.comments.length},timestamp:Date.now(),sessionId:'debug-session',runId:'run1',hypothesisId:'D'})}).catch(()=>{});
      // #endregion
      
      // Preparar comentários para envio (garantir estrutura correta)
      var commentsToSync = state.comments.map(function(comment) {
        return {
          id: String(comment.id || ''),
          text: String(comment.text || ''),
          author: String(comment.author || 'Usuário'),
          timestamp: parseFloat(comment.timestamp || 0),
          duration: parseFloat(comment.duration || 0)
        };
      });

      logDebug('Comentários preparados: ' + commentsToSync.length, 'info');
      if (commentsToSync.length > 0) {
        logDebug('Primeiro comentário preparado: timestamp=' + commentsToSync[0].timestamp + 's', 'info');
      }

      // Preparar JSON string escapado corretamente para ExtendScript
      var commentsJson = JSON.stringify(commentsToSync);
      // Escapar aspas e quebras de linha para uso dentro de string ExtendScript
      var escapedJson = commentsJson.replace(/\\/g, '\\\\').replace(/"/g, '\\"').replace(/\n/g, '\\n').replace(/\r/g, '\\r');
      
      // #region agent log
      fetch('http://127.0.0.1:7242/ingest/1c8fff22-65c6-4471-824d-a58282891d16',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({location:'main.js:744',message:'JSON preparado para ExtendScript',data:{originalLength:commentsJson.length,escapedLength:escapedJson.length,firstChars:commentsJson.substring(0,100)},timestamp:Date.now(),sessionId:'debug-session',runId:'run2',hypothesisId:'C'})}).catch(()=>{});
      // #endregion
      
      var script = 'if (typeof $.global.FlluirSync !== "undefined" && typeof $.global.FlluirSync.createMarkers === "function") { ' +
                   '  try { ' +
                   '    var commentsData = "' + escapedJson + '"; ' +
                   '    $.global.FlluirSync.createMarkers(commentsData); ' +
                   '  } catch (e) { ' +
                   '    JSON.stringify({success: false, error: "Erro ao executar createMarkers: " + e.toString()}); ' +
                   '  } ' +
                   '} else { ' +
                   '  JSON.stringify({success: false, error: "FlluirSync.createMarkers não está disponível"}); ' +
                   '}';

      logDebug('Executando script de criação de marcadores...', 'info');
      
      // #region agent log
      fetch('http://127.0.0.1:7242/ingest/1c8fff22-65c6-4471-824d-a58282891d16',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({location:'main.js:750',message:'Antes de executar createMarkers',data:{commentsCount:commentsToSync.length,firstComment:commentsToSync[0]},timestamp:Date.now(),sessionId:'debug-session',runId:'run1',hypothesisId:'C'})}).catch(()=>{});
      // #endregion
      
      csInterface.evalScript(script, function(result) {
        logDebug('Resultado bruto recebido: ' + String(result).substring(0, 200), 'info');
        
        // #region agent log
        fetch('http://127.0.0.1:7242/ingest/1c8fff22-65c6-4471-824d-a58282891d16',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({location:'main.js:756',message:'Resultado createMarkers recebido',data:{rawResult:result?String(result).substring(0,500):null},timestamp:Date.now(),sessionId:'debug-session',runId:'run1',hypothesisId:'C'})}).catch(()=>{});
        // #endregion
        
        try {
          if (!result || result.trim() === '') {
            logDebug('Resultado vazio recebido do evalScript', 'error');
            showMessage('Erro: Nenhuma resposta do Premiere Pro', 'error');
            return;
          }

          var data = JSON.parse(result);
          
          // #region agent log
          fetch('http://127.0.0.1:7242/ingest/1c8fff22-65c6-4471-824d-a58282891d16',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({location:'main.js:768',message:'createMarkers resultado parseado',data:data,timestamp:Date.now(),sessionId:'debug-session',runId:'run1',hypothesisId:'C'})}).catch(()=>{});
          // #endregion
          logDebug('=== RESULTADO DA SINCRONIZAÇÃO ===', 'info');
          logDebug('Success: ' + data.success, 'info');
          logDebug('Created: ' + (data.created || 0), 'info');
          logDebug('Total: ' + (data.total || 0), 'info');
          
          if (data.error) {
            showMessage('Erro: ' + data.error, 'error');
            logDebug('Erro: ' + data.error, 'error');
          } else if (data.success) {
            var created = data.created || 0;
            var total = data.total || commentsToSync.length;
            var msg = created === total
              ? 'Sincronizado: ' + created + ' marcador' + (created !== 1 ? 'es' : '') + ' criado' + (created !== 1 ? 's' : '')
              : 'Sincronizado: ' + created + ' de ' + total + ' marcadores criados' + (data.errors ? ' (alguns falharam)' : '');
            showMessage(msg, created === total ? 'success' : 'error');
            logDebug('Sincronização concluída: ' + created + ' de ' + total + ' marcadores', created === total ? 'success' : 'error');
            if (data.errors && data.errors.length > 0) {
              logDebug('Erros encontrados:', 'error');
              data.errors.forEach(function(err) {
                logDebug('  - ' + err, 'error');
              });
            }
          } else {
            showMessage('Erro desconhecido na sincronização', 'error');
            logDebug('Resposta inesperada: ' + JSON.stringify(data), 'error');
          }
        } catch (error) {
          showMessage('Erro ao processar sincronização: ' + error.message, 'error');
          logDebug('Erro ao parsear resultado: ' + error.message, 'error');
          logDebug('Resultado recebido (raw): ' + String(result), 'error');
        }
      });
    }
  }

  // Update status indicator
  function updateStatus(status) {
    if (!elements.statusIndicator) {
      console.warn('updateStatus: statusIndicator não encontrado');
      return;
    }
    elements.statusIndicator.className = 'status-indicator ' + status;
    state.isConnected = (status === 'connected');
  }

  // Update current file display
  function updateCurrentFileDisplay(fileInfo) {
    if (!elements.currentFile) {
      console.warn('updateCurrentFileDisplay: currentFile não encontrado');
      return;
    }
    if (fileInfo && fileInfo.success) {
      elements.currentFile.innerHTML = '<div style="padding: 12px; background: #2a2a2a; border-radius: 4px;">' +
                                       '<div style="font-size: 13px; font-weight: 600; color: #fff; margin-bottom: 4px;">' +
                                       fileInfo.name + '</div>' +
                                       '<div style="font-size: 11px; color: #999;">ID: ' + fileInfo.id + '</div>' +
                                       '</div>';
    } else {
      elements.currentFile.innerHTML = '<div class="empty-state-text">Nenhuma sequência ativa</div>';
    }
  }

  // Render comments list
  function renderComments(comments) {
    if (comments.length === 0) {
      elements.commentsList.innerHTML = '<div class="empty-state">' +
                                       '<div class="empty-state-icon">💬</div>' +
                                       '<div class="empty-state-text">Nenhum comentário encontrado</div>' +
                                       '</div>';
      return;
    }

    var html = '';
    comments.forEach(function(comment) {
      var time = formatTime(comment.timestamp || 0);
      html += '<div class="comment-item" data-id="' + comment.id + '" data-time="' + (comment.timestamp || 0) + '">' +
              '<div class="comment-header">' +
              '<span class="comment-author">' + (comment.author || 'Usuário') + '</span>' +
              '<span class="comment-time">' + time + '</span>' +
              '</div>' +
              '<div class="comment-text">' + (comment.text || '').replace(/\n/g, '<br>') + '</div>' +
              '</div>';
    });

    elements.commentsList.innerHTML = html;

    // Click handlers: pular preview e playhead
    var items = elements.commentsList.querySelectorAll('.comment-item');
    items.forEach(function(item) {
      item.addEventListener('click', function() {
        items.forEach(function(i) { i.classList.remove('active'); });
        item.classList.add('active');
        
        var t = Number(item.dataset.time || 0);
        
        // Pular preview
        if (elements.videoPreview && elements.videoPreview.style.display !== 'none') {
          try {
            elements.videoPreview.currentTime = Math.max(0, t);
            elements.videoPreview.play().catch(function() {});
          } catch (e) {}
        }
        
        // Pular playhead do Premiere
        movePlayheadToTime(Math.max(0, t));
      });
    });
  }

  // Clear all markers from the active sequence in Premiere
  function clearMarkersInPremiere() {
    if (!csInterface) {
      showMessage('CSInterface não inicializado', 'error');
      return;
    }

    // Detect extension path to ensure JSX is loaded (reutiliza lógica do sync)
    var extensionPath = '';
    try {
      extensionPath = csInterface.getSystemPath(csInterface.EXTENSION_PATH || 'extension');
    } catch (e1) {
      try {
        extensionPath = csInterface.getSystemPath('extension');
      } catch (e2) {
        extensionPath = '/Users/vitorpassalia/Library/Application Support/Adobe/CEP/extensions/Plugin Flluir';
      }
    }
    extensionPath = extensionPath.replace(/\\/g, '/').replace(/file:\/\//g, '');
    var jsxPath = extensionPath + '/jsx/FlluirSync.jsx';

    // Garantir que o contexto de scripting está carregado (mas a remoção em si usa um script direto)
    var checkScript = 'try { ' +
                      '  if (typeof app !== "undefined" && app.project && app.project.activeSequence) { ' +
                      '    "OK"; ' +
                      '  } else { ' +
                      '    "NO_SEQUENCE"; ' +
                      '  } ' +
                      '} catch (e) { "ERR"; }';

    elements.clearMarkers.disabled = true;

    csInterface.evalScript(checkScript, function(checkResult) {
      if (checkResult === 'NO_SEQUENCE') {
        showMessage('Nenhuma sequência ativa no Premiere para limpar', 'error');
        elements.clearMarkers.disabled = false;
        return;
      }
      if (checkResult === 'ERR') {
        showMessage('Erro ao acessar a sequência no Premiere', 'error');
        elements.clearMarkers.disabled = false;
        return;
      }

      // Script ExtendScript que remove todos os marcadores da sequência ativa
      var clearScript =
        'try {' +
        '  if (!app || !app.project || !app.project.activeSequence) {' +
        '    JSON.stringify({success:false,error:"Nenhuma sequência ativa"});' +
        '  } else {' +
        '    var seq = app.project.activeSequence;' +
        '    if (!seq.markers) {' +
        '      JSON.stringify({success:true,deleted:0});' +
        '    } else {' +
        '      var markers = seq.markers;' +
        '      var deleted = 0;' +
        '      var m = markers.getFirstMarker();' +
        '      while (m) {' +
        '        var nextM = markers.getNextMarker(m);' +
        '        try { markers.deleteMarker(m); deleted++; } catch (e) {}' +
        '        m = nextM;' +
        '      }' +
        '      JSON.stringify({success:true,deleted:deleted});' +
        '    }' +
        '  }' +
        '} catch (e) {' +
        '  JSON.stringify({success:false,error:e.toString()});' +
        '}';

      csInterface.evalScript(clearScript, function(result) {
        elements.clearMarkers.disabled = false;

        if (!result || typeof result !== 'string') {
          showMessage('Erro ao desfazer marcações: resposta vazia do Premiere', 'error');
          return;
        }

        var data;
        try {
          data = JSON.parse(result);
        } catch (e) {
          showMessage('Erro ao desfazer marcações: não foi possível ler o resultado', 'error');
          return;
        }

        if (!data.success) {
          showMessage('Erro ao desfazer marcações: ' + (data.error || 'Erro desconhecido'), 'error');
          return;
        }

        var deleted = data.deleted || 0;
        if (deleted === 0) {
          showMessage('Nenhum marcador encontrado para remover', 'info');
        } else {
          showMessage(deleted + ' marcador' + (deleted !== 1 ? 'es' : '') + ' removido' + (deleted !== 1 ? 's' : '') + ' da sequência', 'success');
        }
      });
    });
  }

  // Move playhead to time
  function movePlayheadToTime(timeInSeconds) {
    if (!csInterface) return;

    var script = 'if (typeof $.global.FlluirSync !== "undefined") { ' +
                 '  $.global.FlluirSync.movePlayheadToTime(' + timeInSeconds + '); ' +
                 '} else { ' +
                 '  JSON.stringify({error: "FlluirSync não carregado"}); ' +
                 '}';

    csInterface.evalScript(script, function(result) {
      try {
        var data = JSON.parse(result);
        if (data.success) {
          logDebug('Playhead movido para: ' + timeInSeconds + 's', 'success');
        }
      } catch (error) {
        logDebug('Erro ao mover playhead: ' + error.message, 'error');
      }
    });
  }

  // Format time
  function formatTime(seconds) {
    var h = Math.floor(seconds / 3600);
    var m = Math.floor((seconds % 3600) / 60);
    var s = Math.floor(seconds % 60);
    return (h > 0 ? h + ':' : '') + 
           (m < 10 ? '0' : '') + m + ':' + 
           (s < 10 ? '0' : '') + s;
  }

  // Show message
  function showMessage(message, type) {
    try {
      var container = document.getElementById('container');
      if (!container) {
        console.warn('showMessage: container não encontrado, usando console');
        console.log('[' + (type === 'error' ? 'ERRO' : 'SUCESSO') + '] ' + message);
        return;
      }
      
      var messageDiv = document.createElement('div');
      messageDiv.className = type === 'error' ? 'error-message' : 'success-message';
      messageDiv.textContent = message;
      
      container.insertBefore(messageDiv, container.firstChild);
      
      setTimeout(function() {
        if (messageDiv.parentNode) {
          messageDiv.remove();
        }
      }, 5000);
    } catch (error) {
      console.error('Erro ao mostrar mensagem:', error);
      console.log('[' + (type === 'error' ? 'ERRO' : 'SUCESSO') + '] ' + message);
    }
  }

  // Debug logging
  function logDebug(message, type) {
    // Sempre logar no console também
    var consoleMethod = type === 'error' ? 'error' : (type === 'success' ? 'log' : 'log');
    console[consoleMethod]('[Flluir] ' + message);
    
    // Se o debug console estiver disponível, adicionar lá também
    if (!elements.debugConsole) {
      return;
    }
    
    try {
      var logDiv = document.createElement('div');
      logDiv.className = 'debug-log' + (type ? ' ' + type : '');
      logDiv.textContent = new Date().toLocaleTimeString() + ' - ' + message;
      elements.debugConsole.appendChild(logDiv);
      elements.debugConsole.scrollTop = elements.debugConsole.scrollHeight;
    } catch (error) {
      console.error('Erro ao adicionar log ao debug console:', error);
    }
  }

  // Toggle debug console
  function toggleDebug() {
    if (!elements.debugConsole) {
      console.warn('toggleDebug: debugConsole não encontrado');
      return;
    }
    elements.debugConsole.classList.toggle('active');
  }

  // Load JSX script
  function loadJSX() {
    // #region agent log
    fetch('http://127.0.0.1:7242/ingest/1c8fff22-65c6-4471-824d-a58282891d16',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({location:'main.js:874',message:'loadJSX chamado',data:{csInterfaceExists:!!csInterface},timestamp:Date.now(),sessionId:'debug-session',runId:'run1',hypothesisId:'D'})}).catch(()=>{});
    // #endregion
    
    if (!csInterface) {
      logDebug('loadJSX: CSInterface não disponível', 'error');
      return;
    }
    
    logDebug('Verificando se JSX está carregado...', 'info');
    
    // Verificar e carregar JSX se necessário
    // Nota: O JSX deve ser carregado automaticamente pelo Premiere, mas vamos garantir
    var checkScript = 'try { ' +
                      '  if (typeof $.global.FlluirSync !== "undefined") { ' +
                      '    "JSX já disponível"; ' +
                      '  } else { ' +
                      '    // Tentar obter o caminho da extensão de diferentes formas ' +
                      '    var extensionPath = ""; ' +
                      '    try { ' +
                      '      extensionPath = csInterface.getSystemPath(csInterface.EXTENSION_PATH); ' +
                      '    } catch (e1) { ' +
                      '      try { ' +
                      '        extensionPath = csInterface.getSystemPath("EXTENSION"); ' +
                      '      } catch (e2) { ' +
                      '        extensionPath = decodeURIComponent(app.project.path).replace(/[^/]*$/, ""); ' +
                      '      } ' +
                      '    } ' +
                      '    var jsxPath = extensionPath + "/jsx/FlluirSync.jsx"; ' +
                      '    $.evalFile(jsxPath); ' +
                      '    if (typeof $.global.FlluirSync !== "undefined") { ' +
                      '      "JSX carregado: " + jsxPath; ' +
                      '    } else { ' +
                      '      "JSX não disponível após carregar de: " + jsxPath; ' +
                      '    } ' +
                      '  } ' +
                      '} catch (e) { ' +
                      '  "Erro: " + e.toString(); ' +
                      '}';
    
    csInterface.evalScript(checkScript, function(result) {
      logDebug('Resultado do carregamento JSX: ' + result, 'info');
      
      if (result && result.indexOf('Erro') === 0) {
        logDebug('Falha ao carregar JSX: ' + result, 'error');
        showMessage('Erro ao carregar JSX. Verifique os logs.', 'error');
      } else if (result && result.indexOf('não disponível') !== -1) {
        logDebug('JSX não está disponível após tentativa de carregamento', 'error');
        showMessage('JSX não carregado. Verifique se o arquivo jsx/FlluirSync.jsx existe.', 'error');
      } else {
        logDebug('JSX verificado/carregado com sucesso', 'success');
      }
      
      // Atualizar sequência ativa
      setTimeout(function() {
        getActiveSequenceInfo().then(function(seq) {
          state.currentFile = seq;
          updateCurrentFileDisplay(seq);
        }).catch(function() {
          updateCurrentFileDisplay(null);
        });
      }, 300);
    });
  }

  // Seções recolhíveis: estado salvo em localStorage
  var COLLAPSED_KEY = 'flluir_collapsed_sections';
  var DEFAULT_COLLAPSED = { config: true, sequence: false, projects: false, gallery: false, comments: false };

  function loadCollapsedState() {
    try {
      var raw = localStorage.getItem(COLLAPSED_KEY);
      if (raw) {
        var parsed = JSON.parse(raw);
        return Object.assign({}, DEFAULT_COLLAPSED, parsed);
      }
    } catch (e) {}
    return Object.assign({}, DEFAULT_COLLAPSED);
  }

  function saveCollapsedState(state) {
    try {
      localStorage.setItem(COLLAPSED_KEY, JSON.stringify(state));
    } catch (e) {}
  }

  function setupCollapsibleSections() {
    var collapsed = loadCollapsedState();
    var sections = document.querySelectorAll('.section.collapsible');
    sections.forEach(function(section) {
      var id = section.getAttribute('data-section');
      if (!id) return;
      if (collapsed[id]) {
        section.classList.add('collapsed');
      } else {
        section.classList.remove('collapsed');
      }
      var header = section.querySelector('.section-header');
      if (!header) return;
      header.addEventListener('click', function() {
        section.classList.toggle('collapsed');
        collapsed[id] = section.classList.contains('collapsed');
        saveCollapsedState(collapsed);
      });
    });
  }

  // Event listeners serão configurados na função init()

  // Initialize
  function init() {
    try {
      console.log('=== INICIANDO PLUGIN FLLUIR SYNC ===');
      
      // Primeiro: vincular elementos do DOM
      if (!bindElements()) {
        var errorMsg = 'Erro: Não foi possível encontrar elementos do DOM. Verifique se o HTML está correto.';
        console.error(errorMsg);
        document.body.innerHTML = '<div style="padding: 20px; color: red; font-family: Arial;">' + errorMsg + '</div>';
        return;
      }
      
      logDebug('Inicializando plugin...', 'info');
      
      // Segundo: inicializar CSInterface
      if (!initCSInterface()) {
        var errorMsg = 'Erro ao inicializar CSInterface. Verifique se o CSInterface.js está presente.';
        console.error(errorMsg);
        showMessage(errorMsg, 'error');
        return;
      }

      // Terceiro: carregar configuração
      loadConfig();
      updateStatus('error');
      resetVideoPreview();
      
      // Quarto: carregar JSX
      loadJSX();
      
      // Quinto: configurar event listeners
      try {
        elements.saveConfig.addEventListener('click', saveConfig);
        elements.testConnection.addEventListener('click', testConnection);
        elements.refreshProjects.addEventListener('click', function() { loadProjects({}); });
        elements.refreshFiles.addEventListener('click', function() {
          if (state.selectedProjectId) {
            loadFiles(state.selectedProjectId, {});
          }
        });
        elements.openFile.addEventListener('click', loadComments);
        elements.loadComments.addEventListener('click', loadComments);
        elements.syncToPremiere.addEventListener('click', syncToPremiere);
        if (elements.clearMarkers) {
          elements.clearMarkers.disabled = false;
          elements.clearMarkers.addEventListener('click', clearMarkersInPremiere);
        }
        elements.toggleDebug.addEventListener('click', toggleDebug);
        console.log('Event listeners configurados com sucesso');
      } catch (err) {
        console.error('Erro ao configurar event listeners:', err);
        logDebug('Erro ao configurar event listeners: ' + err.message, 'error');
      }

      // Seções recolhíveis (Configuração, Projetos, Galeria, Comentários)
      setupCollapsibleSections();
      
      // Sexto: mostrar sequência ativa (não bloqueante)
      getActiveSequenceInfo().then(function(seq) {
        state.currentFile = seq;
        updateCurrentFileDisplay(seq);
      }).catch(function(err) {
        logDebug('Não foi possível obter sequência: ' + (err.message || err), 'error');
        updateCurrentFileDisplay(null);
      });

      logDebug('Plugin Flluir Sync (CEP) inicializado com sucesso', 'success');
      console.log('=== PLUGIN INICIALIZADO COM SUCESSO ===');
      
      // Auto carregar se tiver config
      if (state.apiUrl && state.authToken) {
        testConnection().catch(function(err) {
          logDebug('Auto-connect falhou: ' + (err.message || err), 'error');
        });
      }
    } catch (error) {
      console.error('=== ERRO FATAL AO INICIALIZAR PLUGIN ===');
      console.error('Erro:', error);
      console.error('Stack:', error.stack);
      logDebug('Erro fatal: ' + error.message, 'error');
      
      // Tentar mostrar erro na tela
      try {
        var container = document.getElementById('container');
        if (container) {
          container.innerHTML = '<div class="error-message" style="padding: 20px; color: red; font-family: Arial;">Erro fatal ao inicializar: ' + error.message + '<br><br>Verifique o console para mais detalhes.</div>';
        } else {
          document.body.innerHTML = '<div style="padding: 20px; color: red; font-family: Arial;">Erro fatal ao inicializar: ' + error.message + '</div>';
        }
      } catch (displayError) {
        console.error('Não foi possível exibir mensagem de erro:', displayError);
      }
    }
  }

  // Start when DOM is ready
  try {
    if (document.readyState === 'loading') {
      document.addEventListener('DOMContentLoaded', init);
    } else {
      init();
    }
  } catch (error) {
    console.error('Erro ao iniciar plugin:', error);
    alert('Erro ao iniciar plugin Flluir Sync: ' + error.message);
  }
})();
